/** @file   civilian.cpp
 * @brief   Implementation of Civial - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "civilian.h" // class's header file
#include "settings.h"
#include "gameanims.h"
#include "GfxManager.h"
#include "www_map.h"
#include "civiliancontroller.h"
#include "redrawqueue.h"
#include "animplayer.h"
#include "soundsamples.h"
#include "warglobals.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Constructors and destructor
/// ===========================

/** Constructor
 */
Civilian::Civilian( CivType ct ) :
  GameObject(),
  m_civtype( ct )
{
  this->initCivilian();
  
  this->setController( new CivilianController( this ) );
}



Civilian::Civilian( CivType ct, const std::string& routename ) :
  GameObject(),
  m_civtype( ct )
{
  this->initCivilian();
  CivilianController* pC = new CivilianController( this );
  pC->setRoute( WarGlobals::pLevel->getRoute( routename ) );
  this->setController( pC );
}



/** Destructor.
 */
Civilian::~Civilian()
{
	
}



///
/// Methods
/// =======

/** updates this object
 */
void Civilian::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      this->state( GameObject::STATE_KILLED );
      this->setCorrectAnim( GameAnims::EKilled );
    }
    return;
  }
  
  // Update the controller and follow its instructions
  BaseController* pC = this->getController();
  pC->update();
  
  this->changeAngle( pC->turn() );
  if ( pC->forward() == 2 ) {
    Vec2D mvec( m_movementVec );
    mvec.rotate( this->angle() );
    this->move( mvec );
  } else if ( pC->forward() == 1 ) {
    Vec2D mvec( m_movementVec );
    mvec.rotate( this->angle() );
    mvec /= 2;
    this->move( mvec );
  }
}



/** Makes sound
 */
void Civilian::makeSound( GameObject::SoundID id ) const
{
  if ( m_civtype == 2 ) {
    // Female sounds
    switch ( id ) {
      case ( GameObject::SND_DIE ): {
        Sound::playSample( SMP_WOMAN_DIE, false );
        break;
      }
    }
    
  } else {
    // Male sounds
    switch ( id ) {
      case ( GameObject::SND_DIE ): {
        Sound::playSample( SMP_CIVILIAN_DIE, false );
        break;
      }
      case ( GameObject::SND_PAIN ): {
        Sound::playSample( SMP_PLAYER_PAIN, false );
        break;
      }
    }
  }
}



/** Kills this civilian
 */
void Civilian::kill()
{
  if ( this->state() == GameObject::STATE_KILLED ||
       this->state() == GameObject::STATE_DYING ) {
    return;
  }
  
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnim( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



/** Redraws this civilian
 */
void Civilian::redraw( RedrawQueue* pQueue )
{
  Vec2D p( this->position() );
  int alX = static_cast<int>( p.x() ) - Map::scrollX;
  int alY = static_cast<int>( p.y() ) - Map::scrollY;
  
  BITMAP* pS = m_animation.currentFrame().asBitmap();
  alX -= pS->w / 2;
  alY -= pS->h / 2;
  int pri = RedrawQueue::PRI_NORMAL;
  if ( this->state() == GameObject::STATE_KILLED ) {
    pri = RedrawQueue::PRI_KILLED;
  }
  pQueue->addRotatedSprite( pri, alX,alY, itofix( m_angle ), pS );
}



/** Hit by bullet
 */
bool Civilian::hitByBullet( Bullet* pB )
{
  ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(),
                                         12, Color(130,60,10) );
  WarGlobals::pPartManager->addSystem( pP );
  this->causeDamage( pB );
  return true;
}



///
/// Getter methods
/// ==============

/** Tells if we're reloading
 */
bool Civilian::reloading() const
{
  return false;
}



/** Returns the type
 */
ObjectID::Type Civilian::objectType() const
{
  return ObjectID::TYPE_CIVILIAN;
}



/** Inits the civilian
 */
void Civilian::initCivilian()
{
  ObjectID::Type oid = ObjectID::TYPE_CIVILIAN;

  m_movementVec.y( - Settings::floatObjProp(oid, "spd_walk:") );
  this->setCorrectAnim( GameAnims::EWalk );

  this->boundingSphere( Settings::floatObjProp( oid, "bounding_sphere:" ) );
  this->setArmor( Settings::floatObjProp( oid, "armor:" ) );

  // Collision points
  this->setCollisionPoint( 0, Vec2D( -13,-13 ) );
  this->setCollisionPoint( 1, Vec2D(  13,-13 ) );
  this->setCollisionPoint( 2, Vec2D(  13, 13 ) );
  this->setCollisionPoint( 3, Vec2D( -13, 13 ) );
  
}



/** Sets the correct animation based on the type of this civilian
 */
void Civilian::setCorrectAnim( int aAnimId )
{

  const eng2d::Animation* pA = 0;
  switch ( m_civtype ) {
    case ( TYPE_CIV1 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilian1, aAnimId );
      break;
    }
    case ( TYPE_CIV2 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilian2, aAnimId );
      break;
    }
    case ( TYPE_CIV3 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilian3, aAnimId );
      break;
    }
    case ( TYPE_CIV4 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilian4, aAnimId );
      break;
    }
    case ( TYPE_CIV5 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilian5, aAnimId );
      break;
    }
    case ( TYPE_CIV6 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilian6, aAnimId );
      break;
    }
    case ( TYPE_DANCER1 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilianDancer1, aAnimId );
      break;
    }
    case ( TYPE_DANCER2 ): {
      pA = &GameAnims::findAnimation( AnimId::KCivilianDancer2, aAnimId );
      break;
    }
  }
  
  this->setAnimation( *pA, aAnimId );
}


} // end of namespace
